<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Models\Blog;
use App\Models\Category;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;
use CloudinaryLabs\CloudinaryLaravel\Facades\Cloudinary;
use Illuminate\Support\Facades\Log;
use App\Models\SeoField;

class AdminBlogFormComponent extends Component
{
    use WithFileUploads;

    public $title, $content, $image, $category_id, $blogId, $isEdit = false;
    public $meta_title, $meta_description, $meta_keywords, $meta_robot, $canonical_url, $custom_url, $json_ld, $og_title, $og_description, $og_image;
    public $categories;
    public $isLoading = false;
    public $contentImage;
    public $existingImage;

    protected $rules = [
        'title' => 'required|string|max:255',
        'content' => 'required',
        'category_id' => 'required|exists:categories,id',
        'image' => 'nullable|image|max:2048', // Max 2MB for featured image
        'contentImage' => 'nullable|image|max:2048', // Max 2MB for content image
        'meta_title' => 'nullable|string|max:200',
        'meta_description' => 'nullable|string',
        'meta_keywords' => 'nullable|string',
        'meta_robot' => 'nullable|string',
        'canonical_url' => 'nullable|string',
        'custom_url' => 'nullable|string',
        'json_ld' => 'nullable|string',
        'og_title' => 'nullable|string',
        'og_description' => 'nullable|string',
        'og_image' => 'nullable|string',
    ];

    public function mount($id = null)
    {
        $this->categories = Category::all();

        if ($id) {
            $blog = Blog::findOrFail($id);
            $seo_field = SeoField::where('blog_id', $blog->id)->first();

            $this->blogId = $blog->id;
            $this->title = $blog->title;
            $this->content = $blog->content;
            $this->category_id = $blog->category_id;
            $this->existingImage = $blog->image;
            
            $this->meta_title = $seo_field->meta_title ?? '';
            $this->meta_description = $seo_field->meta_description ?? '';
            $this->meta_keywords = $seo_field->meta_keywords ?? '';
            $this->meta_robot = $seo_field->meta_robot ?? '';
            $this->canonical_url = $seo_field->canonical_url ?? '';
            $this->custom_url = $seo_field->custom_url ?? '';
            $this->json_ld = $seo_field->json_ld ?? '';
            $this->og_title = $seo_field->og_title ?? '';
            $this->og_description = $seo_field->og_description ?? '';
            $this->og_image = $seo_field->og_image ?? '';
            
            $this->isEdit = true;
        } else {
            $this->isEdit = false;
        }
    }

    public function updatedContentImage()
    {
        $this->validate(['contentImage' => 'required|image|max:2048']);

        try {
            $url = Cloudinary::upload($this->contentImage->getRealPath(), [
                'folder' => 'blog_content_images',
            ])->getSecurePath();

            $this->dispatch('content-image-uploaded', url: $url);
        } catch (\Exception $e) {
            Log::error('Cloudinary upload failed for content image: ' . $e->getMessage());
            $this->dispatch('summernoteImageUploadError', message: 'Failed to upload image. Please try again.');
        }
    }

    public function save()
    {
        $this->validate();
    
        try {
            $imageUrl = null;
            if ($this->image) {
                Log::info('Uploading featured image to Cloudinary: ' . $this->image->getRealPath());
                $uploaded = Cloudinary::upload($this->image->getRealPath(), [
                    'folder' => 'blog_featured_images',
                    'transformation' => [
                        'quality' => 'auto',
                        'fetch_format' => 'auto'
                    ]
                ]);
                $imageUrl = $uploaded->getSecurePath();
                Log::info('Featured image uploaded successfully: ' . $imageUrl);
            }
    
            $content = $this->content;
    
            if ($this->isEdit) {
                $blog = Blog::findOrFail($this->blogId);
                
                // If we have a new image and there was an old one, we might want to delete the old one, but for now let's just update
                // Optional: Delete old image if new one is uploaded
                /*
                if ($imageUrl && $blog->image) {
                    $this->deleteCloudinaryImage($blog->image);
                }
                */

                $blog->update([
                    'title' => $this->title,
                    'content' => $content,
                    'image' => $imageUrl ?? $blog->image,
                    'category_id' => $this->category_id,
                    'user_id' => auth()->id(),
                ]);

                SeoField::updateOrCreate(
                    ['blog_id' => $blog->id],
                    [
                        'name_route' => 'single-blog',
                        'meta_title' => $this->meta_title,
                        'meta_description' => $this->meta_description,
                        'meta_keywords' => $this->meta_keywords,
                        'meta_robot' => $this->meta_robot,
                        'canonical_url' => $this->canonical_url,
                        'custom_url' => $this->custom_url,
                        'json_ld' => $this->json_ld,
                        'og_title' => $this->og_title,
                        'og_description' => $this->og_description,
                        'og_image' => $this->og_image,
                    ]
                );

                Log::info('Blog updated: ' . $blog->id);
                session()->flash('message', 'Blog updated successfully!');
            } else {
                $blog = Blog::create([
                    'title' => $this->title,
                    'slug' => Str::slug($this->title),
                    'content' => $content,
                    'image' => $imageUrl,
                    'category_id' => $this->category_id,
                    'user_id' => auth()->id(),
                ]);

                SeoField::create([
                    'blog_id' => $blog->id,
                    'name_route' => 'single-blog',
                    'meta_title' => $this->meta_title,
                    'meta_description' => $this->meta_description,
                    'meta_keywords' => $this->meta_keywords,
                    'meta_robot' => $this->meta_robot,
                    'canonical_url' => $this->canonical_url,
                    'custom_url' => $this->custom_url,
                    'json_ld' => $this->json_ld,
                    'og_title' => $this->og_title,
                    'og_description' => $this->og_description,
                    'og_image' => $this->og_image,
                ]);

                Log::info('Blog created: ' . $blog->id);
                session()->flash('message', 'Blog created successfully!');
            }
    
            return redirect()->route('admin-blog');

        } catch (\Exception $e) {
            Log::error('Failed to save blog: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            $this->addError('form', 'Failed to save blog. Please try again or contact support.');
        }
    }

    private function deleteCloudinaryImage($imageUrl)
    {
        $path = parse_url($imageUrl, PHP_URL_PATH);
        $parts = explode('/', $path);
        $publicIdToDelete = null;
        $uploadIndex = array_search('upload', $parts);

        if ($uploadIndex !== false && count($parts) > $uploadIndex + 2) {
            $publicIdWithFolderAndExtension = implode('/', array_slice($parts, $uploadIndex + 2));
            $folderPath = dirname($publicIdWithFolderAndExtension);
            $filename = pathinfo($publicIdWithFolderAndExtension, PATHINFO_FILENAME);
            
            if ($folderPath === '.') {
                $publicIdToDelete = $filename;
            } else {
                $publicIdToDelete = $folderPath . '/' . $filename;
            }
        }

        if ($publicIdToDelete) {
            try {
                Cloudinary::destroy($publicIdToDelete);
                Log::info('Image deleted from Cloudinary: ' . $publicIdToDelete);
            } catch (\Exception $e) {
                Log::error('Failed to delete image from Cloudinary: ' . $e->getMessage());
                session()->flash('error', 'Failed to delete image from Cloudinary: ' . $e->getMessage());
            }
        }
    }

    public function render()
    {
        return view('livewire.admin.admin-blog-form-component')->layout('layouts.adminbase');
    }
}
