<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Blog;
use App\Models\Category;
use App\Models\Comment;
use Illuminate\Support\Str;

class BlogDetailComponent extends Component
{
    public $slug;
    public $name;
    public $email;
    public $comment;
    public $url; // Basic honeypot or website field

    protected $rules = [
        'name' => 'required',
        'email' => 'required|email',
        'comment' => 'required',
    ];

    public function mount($slug)
    {
        $this->slug = $slug;
    }

    public function saveComment()
    {
        $this->validate();

        $blog = Blog::where('slug', $this->slug)->firstOrFail();

        $comment = new Comment();
        $comment->blog_id = $blog->id;
        $comment->name = $this->name;
        $comment->email = $this->email;
        $comment->content = $this->comment;
        // $comment->website = $this->url; 
        $comment->user_id = auth()->id(); // Nullable
        $comment->approved = true; 
        $comment->save();

        session()->flash('success_comment', 'Your comment has been submitted.');
        $this->reset(['name', 'email', 'comment', 'url']);
    }

    public function render()
    {
        $blog = Blog::where('slug', $this->slug)->with(['user', 'category', 'comments' => function($q) {
            $q->where('approved', true)->latest();
        }])->firstOrFail();

        $recentBlogs = Blog::where('id', '!=', $blog->id)->latest()->take(3)->get();
        $categories = Category::withCount('blogs')->get();
        
        // Previous/Next navigation
        $previousBlog = Blog::where('id', '<', $blog->id)->orderBy('id', 'desc')->first();
        $nextBlog = Blog::where('id', '>', $blog->id)->orderBy('id', 'asc')->first();


        // Tags Logic for Cloud
        $allKeywords = Blog::whereNotNull('meta_keywords')->pluck('meta_keywords');
        $tags = collect();
        foreach ($allKeywords as $keywordStr) {
            $split = explode(',', $keywordStr);
            foreach ($split as $tag) {
                $tags->push(trim($tag));
            }
        }
        $tags = $tags->unique()->take(15);

        // JSON-LD Schema
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'BlogPosting',
            'headline' => $blog->meta_title ?? $blog->title,
            'image' => asset($blog->image),
            'datePublished' => $blog->created_at->toIso8601String(),
            'dateModified' => $blog->updated_at->toIso8601String(),
            'author' => [
                '@type' => 'Person',
                'name' => $blog->user->name ?? 'Admin',
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => config('app.name', 'Dumes'),
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => asset('assets/images/logo.svg'), 
                ],
            ],
            'description' => $blog->meta_description ?? Str::limit(strip_tags($blog->content), 160),
            'mainEntityOfPage' => [
                '@type' => 'WebPage',
                '@id' => route('blog.detail', $blog->slug),
            ],
        ];

        return view('livewire.blog-detail-component', [
            'blog' => $blog,
            'recentBlogs' => $recentBlogs,
            'categories' => $categories,
            'previousBlog' => $previousBlog,
            'nextBlog' => $nextBlog,
            'tags' => $tags,
            'schema' => $schema
        ])->layout('components.layouts.base', [
            'title' => $blog->meta_title ?? $blog->title,
            'metaDescription' => $blog->meta_description ?? Str::limit(strip_tags($blog->content), 160),
            'metaKeywords' => $blog->meta_keywords
        ]);
    }
}
